using UnityEngine;

public class ZOSingleton<T> : MonoBehaviour where T : Component
{
	protected static T _instance;
	public static bool HasInstance => _instance != null;

	/// <summary>
	/// Singleton design pattern
	/// </summary>
	/// <value>The instance.</value>
	public static T Instance
	{
		get
		{
			if (_instance == null)
			{
				_instance = FindObjectOfType<T>();
				if (_instance == null)
				{
					GameObject obj = new()
					{
						name = typeof(T).Name + "_AutoCreated"
					};

					_instance = obj.AddComponent<T>();
				}
			}
			return _instance;
		}
	}

	/// <summary>
	/// On awake, we initialize our instance. Make sure to call base.Awake() in override if you need awake.
	/// </summary>
	protected virtual void Awake()
	{
		InitializeSingleton();
	}

	/// <summary>
	/// Initializes the singleton.
	/// </summary>
	protected virtual void InitializeSingleton()
	{
		if (!Application.isPlaying)
		{
			return;
		}

		_instance = this as T;
	}
}